package com.fluentcommerce.util.core;

import com.fluentretail.api.model.attribute.Attribute;
import com.fluentretail.rubix.rule.meta.RuleInfo;
import com.fluentretail.rubix.v2.context.Context;
import com.fluentretail.rubix.v2.rule.Rule;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class LogUtils {

    private static String getLogTitle(final Class<? extends Rule> ruleClass) {
        return String.format("Rule logs for rule %s", ruleClass.getAnnotation(RuleInfo.class).name());
    }

    /**
     * <p>This method will log a single message that will be later accessible by querying events. The title of the log collection
     * will be taken from the rule's name ('@param ruleClass'), and the subtitle will contain the result of "String.format()"
     * with '@param formatString' as the format string, and '@param formatArgs' as the format arguments mentioned in '@param formatString'.</p>
     *
     * @param context the context of the rule.
     * @param ruleClass the class of the rule, from where the rule name will be taken.
     * @param formatString the format string (Uses "String.format()" notation).
     * @param formatArgs the format arguments mentioned in '@param formatString' ("String.format()" arguments).
     */
    public static void logOnce(final Context context, final Class<? extends Rule> ruleClass, final String formatString, final Object... formatArgs) {
        context.action().log(
                getLogTitle(ruleClass),
                String.format(formatString, formatArgs),
                new LinkedList<>()
        );
    }

    /**
     * <p>This method will log a List<<a href="#{@link}">{@link Attribute}</a>> that will be later accessible by querying events.
     * The title of the log collection will be taken from the rule's name ('@param ruleClass'), and the subtitle will contain
     * "LogCollection:[x]" where 'x' is the size fo the '@param attributes'.</p>
     *
     * <p>This method call "context.action().log()", and it's a final operation. To add/append logs use the ContextWrapper methods.</p>
     *
     * @param context the context of the rule.
     * @param ruleClass the class of the rule, from where the rule name will be taken.
     * @param attributes the List<<a href="#{@link}">{@link Attribute}</a>> that will be logged. It must have at least one Attribute.
     */
    public static void logAttributeCollection(final Context context, final Class<? extends Rule> ruleClass, final List<Attribute> attributes) {
        if (!attributes.isEmpty()){
            context.action().log(
                    getLogTitle(ruleClass),
                    String.format("LogCollection:[%s]", attributes.size()),
                    new ArrayList<>(attributes)
            );
        }
    }

}
