package com.fluentcommerce.util.core;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.module.jsonSchema.JsonSchema;
import com.fasterxml.jackson.module.jsonSchema.JsonSchemaGenerator;

import java.io.IOException;
import java.util.Map;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;

/**
 * Utility class for JSON operations using Jackson library.
 */
public class JsonUtils {

    private static final ObjectMapper mapper = new ObjectMapper();
    private static final JsonSchemaGenerator schemaGen = new JsonSchemaGenerator(mapper);

    static {
        mapper.disable(FAIL_ON_UNKNOWN_PROPERTIES);
    }

    /**
     * Converts a Plain Old Java Object (POJO) into a Map<String, Object>.
     * This method uses Jackson's ObjectMapper to perform the conversion,
     * allowing for easy manipulation of the object's properties as key-value pairs.
     *
     * @param pojo The POJO to convert.
     * @return A Map<String, Object> representation of the POJO.
     */
    public static Map<String, Object> pojoToMap(final Object pojo) {
        return mapper.convertValue(pojo, new TypeReference<Map<String, Object>>() {});
    }

    /**
     * Converts a given Java object into a Jackson ObjectNode.
     * This method leverages Jackson's ObjectMapper to transform the object into a tree structure,
     * which can be useful for JSON manipulation and traversal.
     *
     * @param obj The object to convert.
     * @return An ObjectNode representation of the object.
     */
    public static ObjectNode objectToNode(final Object obj) {
        return mapper.valueToTree(obj);
    }

    /**
     * Converts an object to the specified POJO type.
     *
     * <p>Returns the object directly if it's already the target type. Special handling is included for
     * {@code ObjectNode} due to runtime casting limitations. Otherwise, uses a mapper for conversion.</p>
     *
     * @param <T> the target type
     * @param unknown the object to convert
     * @param type the class of the target type
     * @return the converted object
     * @throws IllegalArgumentException if conversion fails
     */
    public static <T> T convertToPojo(final Object unknown, final Class<T> type) {
        if (type.isInstance(unknown)) {
            return (T) unknown;
        }
        // Reason why I did it because we cannot cast or check `instanceof` ObjectNode to ObjectNod in Rubix runtime environment
        if (ObjectNode.class.getName().equals(unknown.getClass().getName())) {
            return stringToPojo(unknown.toString(), type);
        }
        return mapper.convertValue(unknown, type);
    }

    /**
     * Converts a JSON string into a specified POJO type.
     * This method uses Jackson's ObjectMapper to deserialize the JSON string into an object of the given type.
     * It handles potential IOException during the conversion process.
     *
     * @param json The JSON string to convert.
     * @param type The target class type.
     * @return An instance of the specified type.
     */
    public static <T> T stringToPojo(final String json, final Class<T> type) {
        try {
            if (type.isInstance(json)) {
                return (T) json;
            }
            return mapper.readValue(json, type);
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * Converts a JSON string into a Jackson JsonNode.
     * This method uses Jackson's ObjectMapper to parse the JSON string into a tree structure,
     * which can be useful for JSON manipulation and traversal.
     * It handles potential IOException during the parsing process.
     *
     * @param json The JSON string to convert.
     * @return A JsonNode representation of the JSON string.
     */
    public static JsonNode stringToNode(final String json) {
        try {
            return mapper.readTree(json);
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * Generates a JSON schema for a given class.
     * This method uses Jackson's JsonSchemaGenerator to create a schema that describes the structure of the class.
     * It throws a RuntimeException if the schema generation fails.
     *
     * @param type The class for which to generate the schema.
     * @return A JsonSchema object representing the schema of the class.
     * @throws RuntimeException If an error occurs during schema generation.
     */
    public static JsonSchema getSchemaForClass(final Class type) {
        try {
            return schemaGen.generateSchema(type);
        } catch (JsonMappingException e) {
            throw new RuntimeException("Failed to derive schema for class: " + type);
        }
    }

    /**
     * Compares a string with a JsonNode for equality after converting the string to the appropriate type
     * based on the JsonNode's type (e.g., textual, integer, double, boolean).
     * This method returns true if the converted string is equal to the JsonNode, otherwise it returns false.
     *
     * @param convertableString The string to convert and compare.
     * @param node The JsonNode to compare with.
     * @return true if the converted string is equal to the JsonNode, false otherwise.
     */
    public static boolean marshallAndEquals(final String convertableString, final JsonNode node) {
        if(node.isTextual()) {
            return convertableString.equals(node.asText());
        }
        if(node.isInt()) {
            return Integer.valueOf(convertableString).equals(node.asInt());
        }
        if(node.isDouble()) {
            return Double.valueOf(convertableString).equals(node.asDouble());
        }
        if(node.isBoolean()) {
            return Boolean.valueOf(node.asBoolean()).equals(Boolean.valueOf(convertableString));
        }
        return false;
    }

    /**
     * Compares a string with a JsonNode after converting the string to the appropriate type
     * based on the JsonNode's type (e.g., textual, integer, double).
     * This method returns an integer representing the comparison result, similar to the compareTo method in Java.
     *
     * @param convertableString The string to convert and compare.
     * @param node The JsonNode to compare with.
     * @return An integer representing the comparison result.
     */
    public static int marshallAndCompare(final String convertableString, final JsonNode node) {
        if(node.isTextual()) {
            return convertableString.compareTo(node.asText());
        }
        if(node.isInt()) {
            return Integer.valueOf(convertableString).compareTo(node.asInt());
        }
        if(node.isDouble()) {
            return Double.valueOf(convertableString).compareTo(node.asDouble());
        }
        return 0;
    }
}
