package com.fluentcommerce.util.core;

import lombok.NonNull;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

public final class GqlDateUtils {

    private static final SimpleDateFormat GQL_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");

    /**
     * Parses an object into a date.
     * <p>
     * This function supports the java 7 Date object as well as a string in the format yyyy-MM-dd'T'HH:mm:ss.SSS'Z'
     * <p>
     * An example of a valid string is "2021-06-09T09:48:51.222Z"
     *
     * @param date the date to parse
     * @return the date as a java 7 date object
     * @throws IllegalArgumentException if date is null or it is a string that could not be parsed.
     */
    public static Date parseGqlDate(@NonNull Object date) {
        if (date instanceof Date) {
            return (Date) date;
        }
        try {
            return GQL_DATE_FORMAT.parse((String) date);
        } catch (ParseException | ClassCastException e) {
            throw new IllegalArgumentException("Can not parse the given date", e);
        }
    }

    /**
     * This function parse a java 7 Date object to String with format: "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"
     *
     * @param date the date to transform
     * @return the date as a string
     */
    public static String toGqlDate(Date date) {
        return GQL_DATE_FORMAT.format(date);
    }

    /**
     * It gives the date format used by GQL: "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"
     *
     * @return the date format used by GQL.
     */
    public static SimpleDateFormat getFormat() {
        return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
    }
}
