package com.fluentcommerce.util.core;

import com.fluentretail.api.v2.model.Entity;
import com.fluentretail.rubix.event.Event;
import com.fluentretail.rubix.v2.context.Context;
import com.google.common.collect.ImmutableMap;

import java.time.Instant;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

public class EventUtils {

    /**
     * Copy the context event with a new name, typically for forwarding on to a
     * new execution or to a webhook.
     *
     * Inbound scheduled events will automatically be converted to non-scheduled.
     *
     * Any event attributes will be retained.
     *
     * @param context current Rubix execution context
     * @param newEventName name for the new event
     */
    public static Event cloneAndRenameEvent(Context context, String newEventName) {
        return cloneAndRenameEvent(context, newEventName, null);
    }

    /**
     * Copy the context event with a new name, typically for forwarding on to a
     * new execution or to a webhook.
     *
     * Scheduled events will automatically be converted to non-scheduled unless the outbound event has a separate entity type.
     *
     * @param context current Rubix execution context
     * @param newEventName name for the new event
     * @param transformer an optional parameter that takes in a Event.Builder and applies more parameters to it.
     */
    public static Event cloneAndRenameEvent(Context context, String newEventName, Function<Event.Builder, Event.Builder> transformer) {
        Event originalEvent = context.getEvent();
        final Entity entity = context.getEntity();
        final String entitySubtype = entity == null ? originalEvent.getEntitySubtype() : entity.getType();
        Event.Builder builder = originalEvent.toBuilder().name(newEventName).entitySubtype(entitySubtype);
        if (transformer != null) {
            builder = transformer.apply(builder);
        }

        Event event = builder.build();
        boolean scheduledOnOverridden = !Objects.equals(originalEvent.getScheduledOn(), event.getScheduledOn());
        event = fixSchedule(context, event, scheduledOnOverridden);
        return event;
    }

    /**
     * Send another event with the same structure but a new name.
     *
     * Inbound scheduled events will automatically be converted to non-scheduled.
     *
     * Any event attributes will be retained.
     *
     * @param context current Rubix execution context
     * @param newEventName name for the new event
     */
    public static void forwardInboundEventWithNewName(Context context, String newEventName) {
        context.action().sendEvent(cloneAndRenameEvent(context, newEventName));
    }

    /**
     * Send another event with the same structure but a new name and attributes
     *
     * Inbound scheduled events will automatically be converted to non-scheduled.
     *
     * Any event attributes will be retained.
     *
     * @param context current Rubix execution context
     * @param eventName name for the new event
     * @param attributes the attributes to attach to the event
     */
    public static void forwardInlineEventWithAttributes(Context context, String eventName, Map<String, Object> attributes) {
        context.action().sendEvent(cloneAndRenameEvent(context, eventName, (b) -> b.attributes(attributes)));
    }

    /**
     * Send another event with the same structure but a new name.
     *
     * This version of forwardEvent has a transformer which allows you to apply custom transformations to the event
     * before it is sent.
     *
     * If the transformer changes the root entity type, it will be considered a cross-domain event and its scheduledOn
     * will be set so the event exits the current rubix thread.
     *
     * Inbound scheduled events will automatically be converted to non-scheduled. However, if the transformer sets
     * a scheduledOn, it will not get overridden.
     *
     * Any event attributes will be retained.
     *
     * @param context current Rubix execution context
     * @param eventName name for the new event
     * @param transformer the event
     */
    public static void forwardCustomInlineEvent(Context context, String eventName, Function<Event.Builder, Event.Builder> transformer) {
        context.action().sendEvent(cloneAndRenameEvent(context, eventName, transformer));
    }

    /**
     * Schedule an event a specified number of seconds into the future.
     *
     * It's important to note that the action to add this event to the scheduling queue
     * will only be processed at the end of this execution. This means that in rare cases
     * the delay will have already passed and the event will execute immediately (albeit
     * still asynchronously from the current execution).
     *
     * @param context current Rubix execution context
     * @param event event to schedule
     * @param seconds number of seconds to delay the event execution
     */
    public static void scheduleEvent(Context context, Event event, int seconds) {
        context.action().sendEvent(event.toBuilder().scheduledOn(
            Date.from(Instant.now().plusSeconds(seconds))
        ).build());
    }

    /**
     * Schedule an event a specified number of seconds into the future.
     *
     * It's important to note that the action to add this event to the scheduling queue
     * will only be processed at the end of this execution. This means that in rare cases
     * the delay will have already passed and the event will execute immediately (albeit
     * still asynchronously from the current execution).
     *
     * This version of the method takes the current `context.getEvent()` and changes the
     * name, maintaining all other fields.
     *
     * @param context current Rubix execution context
     * @param eventName name for the new event
     * @param seconds number of seconds to delay the event execution
     */
    public static void scheduleEvent(Context context, String eventName, int seconds) {
        scheduleEvent(context, context.getEvent().toBuilder().name(eventName).build(), seconds);
    }

    /**
     * Add an attribute to an existing event.
     *
     * Events are usually immutable so this helps avoid lots of builder logic in rule.
     *
     * @param event any Rubix event
     * @param attributeName name for the new attribute
     * @param attributeValue value for the new attribute
     *
     * @return a new event based on the inbound event but with an additional attribute.
     */
    public static Event appendEventAttribute(Event event, String attributeName, Object attributeValue) {
        return event.toBuilder()
                .attributes(ImmutableMap.<String, Object>builder()
                        .putAll(event.getAttributes())
                        .put(attributeName, attributeValue)
                        .build())
                .build();
    }

    /**
     * Return the named event attribute (if it exists) in the right type.
     *
     * This method will convert between similar (but not identical) types used across
     * different plugins.
     *
     * @param context current Rubix execution context
     * @param name String name of the event attribute
     * @param clazz Class literal of the desired response type
     * @param <T> the desired type of the attribute
     * @return an Optional of the attribute value (if present and of the right the type)
     */
    public static <T> Optional<T> getEventAttributeAs(Context context, String name, Class<T> clazz) {
        Object value = context.getEvent().getAttributes().get(name);
        if(value == null) {
            return Optional.empty();
        }
        return Optional.of(JsonUtils.convertToPojo(value, clazz));
    }

    /**
     * For events outbound for a different entity type that don't already have a set time, we need to set a small delay,
     * so they don't execute inline with other events. This is required as cross workflow rubix rules do not execute
     * within the same event block.
     *
     * Otherwise, they will become non-scheduled events which will be executed inline. This ensures multiple rulesets
     * in one workflow are executed as one block, and actions will not be effected until the whole event flow succeeds.
     * If for example an event in the third ruleset fails, changes in state from the first ruleset will not take effect.
     *
     * @param context the context of the current rubix execution
     * @param event the event to send
     * @param scheduledOnOverridden true if the user has manually overridden scheduledOn
     * @return the event with a fixed schedule
     */
    private static Event fixSchedule(Context context, Event event, boolean scheduledOnOverridden) {
        Event fixedEvent = event;
        if (!fixedEvent.getRootEntityType().equals(context.getEvent().getRootEntityType())) {
            if (fixedEvent.getScheduledOn() == null) {
                fixedEvent = fixedEvent.toBuilder().scheduledOn(new Date()).build();
            }
        } else {
            //Only set scheduledOn to null if the user didn't override scheduledOn themselves
            if (fixedEvent.getScheduledOn() != null && !scheduledOnOverridden) {
                fixedEvent = fixedEvent.toBuilder().scheduledOn(null).build();
            }
        }
        return fixedEvent;
    }
}
